package sampling;

import java.util.Enumeration;
import java.util.NoSuchElementException;
import java.util.Vector;
import misc.Debug;


/**
 * An object of class Watcher has its own thread of control. It administers
 * a set of sampler objects and their event times. It executes the methods
 * of the samplers at the given times.
 *
 * @version 0.10 03 Nov 1996
 * @author <A HREF="http:www.th-darmstadt.de/~mkarsten">Martin Karsten</A>
 */
public class Watcher extends Thread {
	protected EventList event_list;
	boolean stopped = false;

/**
 * Creates a watcher thread.
 */
	public Watcher( String name ) {
		super( name );
		event_list = new EventList();
		this.start();
		Debug.out( "Watcher.Watcher: Watcher created" );
	}

/**
 * The method add() adds a sampler to this Watcher object.
 * @param sampler The sampler that's been added.
 */
	void add( BaseSampler sampler ) {
		event_list.add( sampler );
		this.interrupt();
	}

/**
 * The method remove() removes a sampler to this Watcher object.
 * @param sampler The sampler that's been removed.
 */
	void remove( BaseSampler sampler ) {
		event_list.remove( sampler );
	}

/**
 * Overrides method run() from class Thread. The main routine for this thread.
 */
	public void run() {
		WatcherEvent event;
		long diff_time = 0;
		Debug.out( "Watcher.run: Watcher running" );
 	 	while ( true ) {
			if ( diff_time > 0 ) {
				try { Thread.sleep( diff_time ); } catch ( InterruptedException i ) { ; }
			}
			synchronized( event_list ) {
 				if ( stopped ) {
		break;
				}
	 	 		event = event_list.getFirst();
	 			if ( event != null ) {
	 				diff_time = event.time - System.currentTimeMillis();
		 			if ( diff_time > 0 ) {
		continue;
	 				}
				} else {
		continue;
				}
 		 		switch( event.et ) {
 	 				case WatcherEvent.POLL_EVENT:
		 		 		Debug.outl( "Watcher.run: POLL_EVENT triggered" );
 	 					event.sampler.poll();
 	 					break;
	 	 			case WatcherEvent.DISPLAY_EVENT:
			 	 		Debug.outl( "Watcher.run: DISPLAY_EVENT triggered" );
 	 					event.sampler.display();
 	 					break;
	 	 		}
		 		event_list.reorder();
 		 	} // synchronized
 	 	} // for
 	 	event_list.finish();
 	 	event_list = null;
 	 	Debug.out( "Watcher.run: Watcher ended" );
	} // run

	/**
	 * To end a properly finish a running Watcher.
	 */
	public void finish() {
		synchronized( event_list ) {
			stopped = true;
			event_list.notify();
			Debug.out( "Watcher.finish: notified Threads in EventList" );
		}
	}

	protected void finalize() {
		Debug.outa("Watcher.finalize: called");
		try { super.finalize(); } catch ( Throwable _x ) { ; }
	}
} // end of class Watcher

/**
 * An object of class WatcherEvent represents a poll or display event
 * from a sampler.  This class is not visible outside the package.
 *
 * @version 0.10 03 Nov 1996
 * @author <A HREF="http:www.th-darmstadt.de/~mkarsten">Martin Karsten</A>
 */
class WatcherEvent {
	public static final int POLL_EVENT = 1;
	public static final int DISPLAY_EVENT = 2;
	protected int et;
	protected BaseSampler sampler;
	protected long time;

/**
 * @param et Type of event (poll or display).
 * @param sampler The sampler this event belongs to.
 * @param time Timepoint when the event shall occur.
 */
	protected WatcherEvent( int et, BaseSampler sampler, long time ) {
		this.et = et;
		this.sampler = sampler;
		this.time = time;
	}

	public void finish() {
		sampler = null;
	}

	protected void finalize() {
		Debug.outa("WatcherEvent.finalize: called");
		try { super.finalize(); } catch ( Throwable _x ) { ; }
	}

} // end of class WatcherEvent

/**
 * An object of class EventList is an ordered list of poll/display events
 * from multiple sampler objects. This list is protected for access from
 * multiple threads. This class is not visible outside the package.
 *
 * @version 0.10 03 Nov 1996
 * @author <A HREF="http:www.th-darmstadt.de/~mkarsten">Martin Karsten</A>
 */
class EventList {
	private Vector list;

/**
 * Using insert_ordered(), an event is inserted into the list. This method is
 * not visible outside the class.
 * @param event The event to be inserted.
 */
	private void insert_ordered( WatcherEvent event ) {
		Enumeration e = list.elements();
		int position = 0;
		while ( e.hasMoreElements() ) {
			if ( event.time < ((WatcherEvent)e.nextElement()).time ) {
				break;
			} // if
			position += 1;
		} // for
		list.insertElementAt( event, position );
	}

/**
 * Creates a list of default size.
 */
	protected EventList() {
		list = new Vector( 100, 20 );
	}

/**
 * The method add() inserts the poll and display event of a given sampler
 * into the list. If a Watcher object is waiting for events, it's notified.
 * @param sampler The sampler object.
 */
	protected synchronized void add( BaseSampler sampler ) {
		insert_ordered( sampler.pollEvent );
		insert_ordered( sampler.displayEvent );
		Debug.out( "EventList.add: Sampler added" );
		this.notify();
	}

/**
 * The method remove() removes the poll and display event of a given sampler
 * from the list.
 * @param sampler The sampler object.
 */
	protected synchronized void remove( BaseSampler sampler ) {
		list.removeElement( sampler.pollEvent );
		list.removeElement( sampler.displayEvent );
		Debug.out( "EventList.remove: Sampler removed" );
	}

/**
 * If no event is stored within the list, the thread that calls this method
 * waits until one arrives. Doesn't need to be synchronized, because of
 * synchronize statement in Watcher.run.
 * @returns The next event in the list
 */
	protected WatcherEvent getFirst() {
		WatcherEvent event = null;
		try {
			event = (WatcherEvent)list.firstElement();
		}
		catch ( NoSuchElementException n ) {
			Debug.out( "EventList.getFirst: Waiting for Samplers" );
			try { this.wait(); } catch ( InterruptedException i ) { ; }
			Debug.out( "EventList.getFirst: Sampler arrived" );
			if ( !list.isEmpty() ) {
				event = (WatcherEvent)list.firstElement();
			}
		}
		return event;
	}

/**
 * The method reorder() reorders the list, i.e. the first event is taken and
 * moved to the correct position. Doesn't need to be synchronized, because of
 * synchronize statement in Watcher.run.
 */
	protected void reorder() {
		WatcherEvent event = (WatcherEvent)list.firstElement();
		list.removeElementAt( 0 );
		insert_ordered( event );
	}

	public synchronized void finish() {
		list.removeAllElements();
		list = null;
		Debug.out( "EventList.finish: called" );
	}

	protected void finalize() {
		Debug.outa("EventList.finalize: called");
		try { super.finalize(); } catch ( Throwable _x ) { ; }
	}


} // end of class EventList
